// Rad_Move.js //
// Entity and player movement counterpart //

// Basic movement command. Will move person 'name', 'x' amount of
// Tiles, in a certain direction, facing a certain way. 
function Move(name, tiles, direction_command, face_command)
{
	if (face_command === undefined) {
		if (direction_command == COMMAND_MOVE_NORTH) face_command = COMMAND_FACE_NORTH;
		if (direction_command == COMMAND_MOVE_SOUTH) face_command = COMMAND_FACE_SOUTH;
		if (direction_command == COMMAND_MOVE_EAST) face_command = COMMAND_FACE_EAST;
		if (direction_command == COMMAND_MOVE_WEST) face_command = COMMAND_FACE_WEST;
		if (direction_command == COMMAND_WAIT) face_command = null;
	}
	
	if (!IsObstructed(name, direction_command)) {
		if (face_command != null) QueuePersonCommand(name, face_command, true);
		for (var i = 0; i < tiles; ++i) MovePixels(name, direction_command, GetTileWidth());
	}
}

// This is used to check if a given person can move
// towards a given direction.
function IsObstructed(name, command)
{
	var H = 0, W = 0;
	var X = GetPersonX(name);
	var Y = GetPersonY(name);
	
	switch(command) {
		case COMMAND_MOVE_NORTH: H = -1; break;
		case COMMAND_MOVE_SOUTH: H =  1; break;
		case COMMAND_MOVE_EAST:  W =  1; break;
		case COMMAND_MOVE_WEST:  W = -1; break;
		default: return false; break;
	}

	if (IsPersonObstructed(name, X+W, Y+H)) return true;
	if (IsPersonObstructed(name, X+(W>>1), Y+(H>>1))) return true;
	return false; // just in case
}

// Utilize the pathfinding algorithm to move
// An entity towards a given destination.
function MoveToLocation(name, x, y, wait)
{
	var Path = GetPath(name, -1, -1, 1, x, y);
	MovePath(name, Path, wait);
}

function MovePath(name, path, wait)
{
	if (wait == undefined) wait = false;
	for (var i = 0; i < path.length; ++i) {
		switch (path.charAt(i)) {
			case "E": Move(name, 1, EAST); break;
			case "N": Move(name, 1, NORTH); break;
			case "S": Move(name, 1, SOUTH); break;
			case "W": Move(name, 1, WEST); break;
			case "F": break;
		}
	}
	if (wait) WaitForPerson(name);
}

function ReversePath(path)
{
	var newpath = "";
	var i = path.length;
	while(i--) {
		var c = path.charAt(i);
		switch (c) {
			case "E": newpath += "W"; break;
			case "W": newpath += "E"; break;
			case "S": newpath += "N"; break;
			case "N": newpath += "S"; break;
			case "F": newpath = "F" + newpath; break;
			default: newpath += c; break;
		}
	}
	return newpath;
}

// Blocks other things from happening, but
// It will update anything that would be
// Updateable in a given timeframe.
// EX: Player movement.
function WaitForPerson(name)
{
	var framerate = GetFrameRate();
	SetFrameRate(GetMapEngineFrameRate());
	while (!IsCommandQueueEmpty(name)) {
		RenderMap();
		UpdateMapEngine();
		FlipScreen();
	}
	SetFrameRate(framerate);
}

function MovePixels(name, direction, amount)
{
	var i = amount;
	while(i--) QueuePersonCommand(name, direction, false);
}

function DoesPersonExist(name)
{
	var List = GetPersonList();
	var i = List.length;
	while(i--) if (List[i] == name) return true;
	return false;
}

function SetPersonToTile(name, x, y)
{
	SetPersonX(name, (x>>4)<<4);
	SetPersonY(name, (y>>4)<<4);
}

function SetPersonXY(name, x, y)
{
	SetPersonX(name, (x<<4)+7);
	SetPersonY(name, (y<<4)+7);
}

// This should be sparingly called //
//  x and y are pixel coordinates  //
function GetPersonAt(x, y, layer)
{
	var pList = GetPersonList();
	var i = pList.length;
	while(i--) {
		var name = pList[i];
		var base = GetPersonBase(name);
		var w = base.x2 - base.x1;
		var h = base.y2 - base.y1;
		var px = GetPersonX(name)-w/2;
		var py = GetPersonY(name)-h/2;
		var l = GetPersonLayer(name);
		if (l != layer) continue;
		if (x > px && x < px+w+1 && y > py && y < py+h+1) return name;
	}
	return "";
}

function Wait(msecs)
{
	var time = GetTime()+msecs;
	var framerate = GetFrameRate();
	SetFrameRate(GetMapEngineFrameRate());
	while (time > GetTime()) {
		RenderMap();
		UpdateMapEngine();
		FlipScreen();
	}
	SetFrameRate(framerate);
}

/* Entity Movement Handling */

function SetPersonPath(name, path)
{
	SetPersonValue(name, "path", path);
	SetPersonValue(name, "pixel", 0);
	SetPersonValue(name, "position", 0);
	SetPersonFrameRevert(name, 8);
}

function WalkPath(name)
{
	var path = GetPersonValue(name, "path");
	var p = GetPersonValue(name, "position");

	if (IsCommandQueueEmpty(name)) {
		SetPersonDirection(name, PathPosToDir(name, path, p));
		var c = DirectionToCommand(name);
		if (IsObstructed(name, c)) return;

		RunCurrentPathDirection(name, path, p);
	}
}

function DirectionToCommand(name)
{
	var d = GetPersonDirection(name);
	switch(d) {
		case "north": return COMMAND_MOVE_NORTH; break;
		case "south": return COMMAND_MOVE_SOUTH; break;
		case "east": return COMMAND_MOVE_EAST; break;
		case "west": return COMMAND_MOVE_WEST; break;
		default: return COMMAND_WAIT; break;
	}
}

function PathPosToDir(name, path, pos)
{
	switch (path.charAt(pos)) {
		case "N": return "north"; break;
		case "S": return "south"; break;
		case "E": return "east"; break;
		case "W": return "west"; break;
		case "R": return PathPosToDir(name, path, 0); break;
		case "P": return PathPosToDir(name, path, pos+1); break;
		default: return GetPersonDirection(name); break;
	}
}

function RunCurrentPathDirection(name, path, pos)
{
	var p = GetPersonValue(name, "pixel")+1;
	switch(path.charAt(pos)) {
		case "N": QueuePersonCommand(name, NORTH, false); break;
		case "S": QueuePersonCommand(name, SOUTH, false); break;
		case "E":	QueuePersonCommand(name, EAST, false); break;
		case "W": QueuePersonCommand(name, WEST, false); break;
		case "P": QueuePersonCommand(name, WAIT, false); break;
		//case "F": SetPersonPath(name, ReversePath(path)); pos=-1; p=16; break;
		case "R": pos=-1; p=16; break;
		default: break; //nothing
	}
	SetPersonValue(name, "pixel", p);
	if (p == 16) {
		SetPersonValue(name, "position", pos+1);
		SetPersonValue(name, "pixel", 0);	
	}
}